<?php

namespace App\Http\Controllers\Backend;

use App\Models\Book;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Crypt;
use App\Http\Requests\StoreBookRequest;
use App\Http\Requests\UpdateBookRequest;
use App\Repository\Backend\Interfaces\BookRepositoryInterface;

class BookController extends Controller
{
    /** @var BookRepositoryInterface */
    private $bookRepository;
    public function __construct(BookRepositoryInterface $bookRepo)
    {
        $this->bookRepository = $bookRepo;
        $this->middleware('permission:book_list', ['only' => ['index','show']]);
        $this->middleware('permission:book_create', ['only' => ['create','store']]);
        $this->middleware('permission:book_edit', ['only' => ['edit','update']]);
        $this->middleware('permission:book_delete', ['only' => ['destroy']]);
        $this->middleware('permission:book_status', ['only' => ['changeStatus']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $paginate=10;
        $order_by=$request->order_by??'books.id';
        $direction=$request->direction??'desc';
        // dd($request->all());
        $input = $request->all();
        $input['order_by']=$order_by;
        $input['direction']=$direction;
        $input['paginate']=$paginate;
        $books = $this->bookRepository->all($input);
        return view('backend.book.index',compact('books','order_by','direction'))->withInput($input);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.book.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreBookRequest $request)
    {
        // dd($request->all());
        $input=$request->all();
        $this->bookRepository->store($input);
        return redirect('admin/book')->with('success', 'New book successfully added');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $id=\Crypt::decrypt($id);
        $book=Book::findorfail($id);
        return view('backend.book.show',compact('book'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $id=\Crypt::decrypt($id);
        $book=Book::findorfail($id);
        return view('backend.book.edit',compact('book'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateBookRequest $request, $id)
    {
        $id=\Crypt::decrypt($id);
        $input=$request->all();
        $book=$this->bookRepository->update($input,$id);
        return redirect()->route('book.index')->with('success', 'Successfully Updated')->withInput($input);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try{
            DB::beginTransaction();
            $id=\Crypt::decrypt(request()->id);
            $book=Book::findorfail($id);

            if($book->file_url)
            {
                Book::deleteImage($book->file_url);
            }
            $delete=$book->delete();
            save_logs('Delete Book',$book);
            DB::commit();
            return redirect('admin/book')->with(['success'=>'Book information successfully delete']);
        } catch (\Illuminate\Database\QueryException $e) {
            DB::rollBack();
            return redirect('admin/book')->with(['fail' => 'Error deleting record'], 500);


        }

    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function changeStatus(Request $request)
    {
        $id=\Crypt::decrypt($request->id);
        $book=Book::findorfail($id);
        $book->status=$request->status=='false'?'inactive':'active';
        $book->save();

        return response()->json(['success'=>'Successfully change status.'],200);
    }

}
