<?php

namespace App\Http\Controllers\Backend;

use File;
use App\Models\Post;
use App\Models\PostFile;
use App\Models\MenuItems;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Http\Requests\StorePostRequest;
use App\Http\Requests\UpdatePostRequest;
use App\Repository\Backend\Interfaces\PostRepositoryInterface;

class PostController extends Controller
{
    /** @var PostRepositoryInterface */
    private $postRepository;
    public function __construct(PostRepositoryInterface $postRepo)
    {
        $this->postRepository = $postRepo;
        $this->middleware('permission:post_list', ['only' => ['index', 'show']]);
        $this->middleware('permission:post_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:post_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:post_delete', ['only' => ['destroy']]);
        $this->middleware('permission:post_status', ['only' => ['changeStatus']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        // session()->forget('post_files');
        $paginate = 10;
        $order_by = $request->order_by ?? 'posts.id';
        $direction = $request->direction ?? 'desc';
        $input = $request->all();
        $menu_item_id = $request->menu_item_id;
        $postMenus = MenuItems::select('id', 'label', 'label_mm')
            ->where('type', 'post')
            ->orwhere('type', 'all')->get();
        $input['order_by'] = $order_by;
        $input['direction'] = $direction;
        $input['paginate'] = $paginate;
        $posts = $this->postRepository->all($input);
        return view('backend.post.index', compact('posts', 'postMenus', 'menu_item_id', 'order_by', 'direction'))->withInput($input);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //query menu_items (post and all type)
        $post_menus = MenuItems::select('id', 'label', 'label_mm', 'slug')
            ->where('type', 'post')
            ->orwhere('type', 'all')->get();
        return view('backend.post.create', compact('post_menus'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StorePostRequest $request)
    {
        $input = $request->all();
        $this->postRepository->store($input);
        return redirect('admin/post')->with('success', 'New post successfully added');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $id = \Crypt::decrypt($id);
        $post = Post::findorfail($id);

        // $post->body=json_decode($post->body);
        // $post->body_mm=json_decode($post->body_mm);
        return view('backend.post.show', compact('post'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $id = \Crypt::decrypt($id);

        $post = Post::findorfail($id);
        // dd($post);
        //query menu_items (post and all type)
        $post_menus = MenuItems::select('id', 'label', 'label_mm', 'slug')
            ->where('type', 'post')
            ->orwhere('type', 'all')->get();

        return view('backend.post.edit', compact('post', 'post_menus'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdatePostRequest $request, $id)
    {
        $id = \Crypt::decrypt($id);
        $input = $request->all();
        $post = $this->postRepository->update($input, $id);
        return redirect()->route('post.index')->with('success', 'Successfully Updated')->withInput($input);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            DB::beginTransaction();
            $logs = [];
            $id = \Crypt::decrypt(request()->id);
            $post = Post::findorfail($id);

            Post::deleteImage($post->feature_image);

            if ($post->files) {
                foreach ($post->files as $file) {
                    array_push($logs, $file);
                    Post::deleteImage($file->file_url);
                    $file->delete();
                }
            }
            array_push($logs, $post);
            save_logs('Delete Post', $logs);
            $post->delete();
            DB::commit();
            return redirect('admin/post')->with(['success' => 'Post information successfully delete']);
        } catch (\Illuminate\Database\QueryException $e) {
            // throw $e;
            DB::rollBack();
            return redirect('admin/post')->with(['fail' => 'Error deleting record'], 500);
        }
    }

    public function changeStatus(Request $request)
    {
        $id = \Crypt::decrypt($request->id);
        $post = Post::findorfail($id);
        $post->status = $request->status == 'false' ? 'inactive' : 'active';
        $post->updated_by = Auth::id() == 1?2:Auth::id();
        $post->save();
        save_logs('Change Post Status', $post);
        return response()->json(['success' => 'Successfully change status.'], 200);
    }
    public function fileDestroy(Request $request)
    {
        try {
            $file = PostFile::findorfail($request->id);
            Post::deleteImage($file->file_url);
            $file->delete();
            save_logs('Delete Post File', $file);
            return response()->json(['message' => 'Successfully destroy']);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['message' => $e->getMessage()]);
        }
    }
    public function getDecryptId(Request $request)
    {
        return response()->json(['id' => \Crypt::decrypt($request->id)]);
    }
    // public function fileUpload(Request $request)
    // {
    //     $filePath = Post::makeImage($request->upload, Post::IMAGE_PATH);
    //     $ckeditorFunNum=$request->input('CKEditorFuncNum');
    //     $url=asset('storage/'.$filePath);
    //     $message="Your Photo Uploaded";
    //     $res="<script>
    //             window.parent.CKEDITOR.tools.callFunction($ckeditorFunNum,`$url`,`$message`)
    //         </script>";
    //         @header("Content-Type:text/html; charset=utf-8");

    //     //save name in session to delete images
    //     $file_array=explode('/',$filePath);
    //     $file_name=end($file_array);
    //     if(session()->has('post_files')){
    //         $files[]=session()->get('post_files');
    //         array_push($files,$file_name);
    //         session()->put('post_files',$files);
    //     }else{

    //         session()->put('post_files',$file_name);
    //     }
    //     echo $res;
    // }


}
