<?php

namespace App\Http\Controllers\Backend;

use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreRoleRequest;
use App\Http\Requests\UpdateRoleRequest;
use Spatie\Permission\Models\Permission;
use App\Repository\Backend\Interfaces\RoleRepositoryInterface;
use Illuminate\Support\Facades\DB;

class RoleController extends Controller
{

    /** @var RoleRepositoryInterface */
    private $roleRepository;
    public function __construct(RoleRepositoryInterface $roleRepo)
    {
        $this->roleRepository = $roleRepo;
        $this->middleware('permission:role_list', ['only' => ['index','show']]);
        $this->middleware('permission:role_create', ['only' => ['create','store']]);
        $this->middleware('permission:role_edit', ['only' => ['edit','update']]);
        $this->middleware('permission:role_delete', ['only' => ['destroy']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $paginate=10;
        $order_by=$request->order_by??'roles.id';
        $direction=$request->direction??'desc';
        $input = $request->all();
        $input['order_by']=$order_by;
        $input['direction']=$direction;
        $input['paginate']=$paginate;
        $roles = $this->roleRepository->all($input);
        return view('backend.role.index',compact('roles','order_by','direction'))->withInput($input);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $permissions = Permission::all()
            // ->sortBy('name', SORT_NATURAL|SORT_FLAG_CASE)
            ->groupBy(function($permission){
                return explode('_', $permission->name)[0];
            });
        return view('backend.role.create',compact('permissions'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreRoleRequest $request)
    {
        $input['name']=$request->name;
        $input['permissions']=$request->permissions;
        $this->roleRepository->store($input);
        return redirect('admin/role')->with('success', 'New role successfully added.');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $id=\Crypt::decrypt($id);
        $role=Role::findorfail($id);
        $role_permissions=$role->permissions->groupBy(function($permission){
            return explode('_', $permission->name)[0];
        });
        return view('backend.role.show',compact('role','role_permissions'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $role=Role::findorfail(\Crypt::decrypt($id));
        $permissions = Permission::all()
            // ->sortBy('name', SORT_NATURAL|SORT_FLAG_CASE)
            ->groupBy(function($permission){
                return explode('_', $permission->name)[0];
            });

        $rolePermissions = [];
        foreach ($role->permissions as $value) {
            array_push($rolePermissions, $value->name);
        }
        return view('backend.role.edit', compact('role', 'permissions', 'rolePermissions'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateRoleRequest $request, $id)
    {
        $id=\Crypt::decrypt($id);
        $input['name']=$request->name;
        $input['permissions']=$request->permissions;
        $role=$this->roleRepository->update($input,$id);
        return redirect()->route('role.index')->with('success', 'Successfully Updated.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try{
            DB::beginTransaction();
            $id=\Crypt::decrypt(request()->id);
            $role=Role::findorfail($id);

            if ($role->users()->exists()) {
                return redirect('admin/role')->with(['fail' => 'Cannot delete this record! This record is related to others.'], 400);
            }else{
                $delete=$role->delete();
            }
            save_logs('Delete Role',$role);
            DB::commit();
            return redirect('admin/role')->with(['success'=>'Role information successfully delete.']);
        } catch (\Illuminate\Database\QueryException $e) {
            DB::rollBack();
            return redirect('admin/role')->with(['fail' => 'Error deleting record!'], 500);

        }
    }
}
