<?php

namespace App\Http\Controllers\Backend;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use App\Http\Requests\StoreUserRequest;
use App\Http\Requests\UpdateUserRequest;
use App\Http\Requests\UpdateProfileRequest;
use App\Repository\Backend\Interfaces\UserRepositoryInterface;

class UserController extends Controller
{
    /** @var UserRepositoryInterface */
    private $userRepository;
    public function __construct(UserRepositoryInterface $userRepo)
    {
        $this->userRepository = $userRepo;
        $this->middleware('permission:user_list', ['only' => ['index', 'show']]);
        $this->middleware('permission:user_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:user_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:user_delete', ['only' => ['destroy']]);
        $this->middleware('permission:user_status', ['only' => ['changeStatus']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $paginate = 10;
        $order_by = $request->order_by ?? 'users.id';
        $direction = $request->direction ?? 'desc';
        // dd($request->all());
        $input = $request->all();
        $input['order_by'] = $order_by;
        $input['direction'] = $direction;
        $input['paginate'] = $paginate;
        $users = $this->userRepository->all($input);
        return view('backend.user.index', compact('users', 'order_by', 'direction'))->withInput($request->all());
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $roles = Role::all();
        return view('backend.user.create', compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreUserRequest $request)
    {
        // dd($request->all());
        $input = $request->all();
        $this->userRepository->store($input);
        return redirect('admin/user')->with('success', 'New user successfully added');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $id = Crypt::decrypt($id);
        $user = User::findorfail($id);
        return view('backend.user.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $id = Crypt::decrypt($id);
        $user = User::findorfail($id);
        $roles = Role::all();
        return view('backend.user.edit', compact('roles', 'user'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateUserRequest $request, $id)
    {
        $id = Crypt::decrypt($id);
        $input = $request->all();
        $user = $this->userRepository->update($input, $id);
        return redirect()->route('user.index')->with('success', 'Successfully Updated')->withInput($input);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            DB::beginTransaction();
            $id = Crypt::decrypt(request()->id);
            $user = User::findorfail($id);

            if (isset($user->profile_image)) {
                User::deleteImage($user->profile_image);
            }
            $delete = $user->delete();
            save_logs('Delete User', $user);
            DB::commit();
            return redirect('admin/user')->with(['success' => 'User information successfully delete']);
        } catch (\Illuminate\Database\QueryException $e) {
            DB::rollBack();
            return redirect('admin/user')->with(['fail' => 'Error deleting record'], 500);
        }
    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function changeStatus(Request $request)
    {
        $id = Crypt::decrypt($request->id);
        $user = User::findorfail($id);
        $user->status = $request->status == 'false' ? 'inactive' : 'active';
        $user->updated_by = Auth::id() == 1?2:Auth::id();
        $user->save();

        return response()->json(['success' => 'Successfully change status.'], 200);
    }

    public function profile($id)
    {
        // dd(auth()->user()->roles);
        $id = Crypt::decrypt($id);
        $user = User::findorfail($id);
        $roles = Role::all();
        return view('backend.user.profile', compact('roles', 'user'));
    }
    public function profileUpdate(UpdateProfileRequest $request, $id)
    {
        $id = Crypt::decrypt($id);
        $input = $request->all();
        $user = $this->userRepository->update($input, $id);
        return redirect()->route('admin-dashboard')->with('success', 'Successfully Updated')->withInput($input);
    }
}
