<?php

namespace App\Http\Controllers\Backend;

use App\Models\Video;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Crypt;
use App\Http\Requests\StoreVideoRequest;
use App\Http\Requests\UpdateVideoRequest;
use App\Repository\Backend\Interfaces\VideoRepositoryInterface;

class VideoController extends Controller
{
    /** @var VideoRepositoryInterface */
    private $videoRepository;
    public function __construct(VideoRepositoryInterface $videoRepo)
    {
        $this->videoRepository = $videoRepo;
        $this->middleware('permission:video_list', ['only' => ['index','show']]);
        $this->middleware('permission:video_create', ['only' => ['create','store']]);
        $this->middleware('permission:video_edit', ['only' => ['edit','update']]);
        $this->middleware('permission:video_delete', ['only' => ['destroy']]);
        $this->middleware('permission:video_status', ['only' => ['changeStatus']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $paginate=10;
        $order_by=$request->order_by??'videos.id';
        $direction=$request->direction??'desc';
        // dd($request->all());
        $input = $request->all();
        $input['order_by']=$order_by;
        $input['direction']=$direction;
        $input['paginate']=$paginate;
        $videos = $this->videoRepository->all($input);
        return view('backend.video.index',compact('videos','order_by','direction'))->withInput($input);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.video.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreVideoRequest $request)
    {
        // dd($request->all());
        $input=$request->all();
        $this->videoRepository->store($input);
        return redirect('admin/video')->with('success', 'New video successfully added');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $id=\Crypt::decrypt($id);
        $video=Video::findorfail($id);
        return view('backend.video.show',compact('video'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $id=\Crypt::decrypt($id);
        $video=Video::findorfail($id);
        return view('backend.video.edit',compact('video'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateVideoRequest $request, $id)
    {
        $id=\Crypt::decrypt($id);
        $input=$request->all();
        $video=$this->videoRepository->update($input,$id);
        return redirect()->route('video.index')->with('success', 'Successfully Updated')->withInput($input);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try{
            DB::beginTransaction();
            $id=\Crypt::decrypt(request()->id);
            $video=Video::findorfail($id);

            if($video->file_url)
            {
                Video::deleteImage($video->file_url);
            }
            $delete=$video->delete();
            save_logs('Delete Video',$video);
            DB::commit();
            return redirect('admin/video')->with(['success'=>'Video information successfully delete']);
        } catch (\Illuminate\Database\QueryException $e) {
            DB::rollBack();
            return redirect('admin/video')->with(['fail' => 'Error deleting record'], 500);


        }

    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function changeStatus(Request $request)
    {
        $id=\Crypt::decrypt($request->id);
        $video=Video::findorfail($id);
        $video->status=$request->status=='false'?'inactive':'active';
        $video->save();

        return response()->json(['success'=>'Successfully change status.'],200);
    }

}
