<?php

namespace App\Repository\Backend;

use Exception;
use Illuminate\Container\Container as Application;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Validator;

abstract class BaseRepository
{
    /**
     * @var Model
     */
    protected $model;

    /**
     * @var Application
     */
    protected $app;

    /**
     * @param  Application  $app
     *
     * @throws Exception
     */
    public function __construct(Application $app)
    {
        $this->app = $app;
        $this->makeModel();
    }

    /**
     * Make Model instance
     *
     * @throws Exception
     *
     * @return Model
     */
    public function makeModel()
    {
        $model = $this->app->make($this->model());

        if (! $model instanceof Model) {
            throw new Exception("Class {$this->model()} must be an instance of Illuminate\\Database\\Eloquent\\Model");
        }

        return $this->model = $model;
    }

    /**
     * Configure the Model
     *
     * @return string
     */
    abstract public function model();

    /**
     * Paginate records for scaffold.
     *
     * @param  int  $perPage
     * @param  array  $columns
     *
     * @return LengthAwarePaginator
     */
    public function paginate($perPage, $columns = ['*'])
    {
        $query = $this->allQuery();

        return $query->paginate($perPage, $columns);
    }

    /**
     * Build a query for retrieving all records.
     *
     * @param  array  $search
     *
     * @return Builder
     */
    public function allQuery($search = [])
    {
        $query = $this->model->newQuery();
        if (count($search)) {
            $searchableFields = $this->getFieldsSearchable();

            foreach ($search as $key => $value) {
                if (in_array($key, $searchableFields)) {
                    $query->where($key,'like', '%'.$value.'%');
                }
            }
            if (! empty($search['search'])) {
                $query = filterByColumns($query, $search['search'], $searchableFields);
            }

            if (! empty($search['order_by'])) {
                $direction = (! empty($search['direction'])) ? $search['direction'] : 'asc';
                $query->orderBy($search['order_by'], $direction);
            }
        }

        return $query;
    }

    /**
     * Get searchable fields array
     *
     * @return array
     */
    abstract public function getFieldsSearchable();

    /**
     * Retrieve all records with given filter criteria
     *
     * @param  array  $search
     *
     * @return LengthAwarePaginator|Builder[]|Collection
     */
    public function all($search = [])
    {
        $query = $this->allQuery($search);

        $result = $query->get();

        return $result;
    }

    /**
     * Create model record
     *
     * @param  array  $input
     *
     * @return Model
     */
    public function create($input)
    {
        $model = $this->model->newInstance($input);

        $model->save();

        return $model;
    }

    /**
     * Find model record for given id
     *
     * @param  int  $id
     * @param  array  $columns
     *
     * @return Builder|Builder[]|Collection|Model|null
     */
    public function find($id, $columns = ['*'])
    {
        $query = $this->model->newQuery();

        return $query->find($id, $columns);
    }

    /**
     * Update model record for given id
     *
     * @param  array  $input
     * @param  int  $id
     *
     * @return Builder|Builder[]|Collection|Model
     */
    public function update($input, $id)
    {
        $query = $this->model->newQuery();

        $model = $query->findOrFail($id);

        $model->fill($input);

        $model->save();

        return $model;
    }

    /**
     * @param  int  $id
     *
     * @throws Exception
     *
     * @return bool|mixed|null
     */
    public function delete($id)
    {
        $query = $this->model->newQuery();

        $model = $query->findOrFail($id);

        return $model->delete();
    }

    /**
     * @param  int  $id
     * @param  array  $with
     * @return mixed
     */
    public function findOrFail($id, $with = [])
    {
        if (! empty($with)) {
            $record = $this->model::with($with)->find($id);
        } else {
            $record = $this->model::find($id);
        }
        if (empty($record)) {
            throw new ModelNotFoundException(class_basename($this->model).' not found.');
        }

        return $record;
    }

    /**
     * @param $request
     * @param $rules
     *
     * @param  array  $ruleMessage
     * @return string
     */
    public function validateRules($request, $rules, $ruleMessage = [])
    {
        $validator = Validator::make($request, $rules, $ruleMessage);
        if ($validator->fails()) {
            return $validator->messages()->first();
        }

        return null;
    }
}
