<?php
namespace App\Repository\Backend;

use App\Models\Book;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Repository\Backend\BaseRepository;
use App\Repository\Backend\Interfaces\BookRepositoryInterface;

class BookRepository extends BaseRepository implements BookRepositoryInterface
{
    /**
     * @var array
     */
    protected $fieldSearchable = [
        'title',
        'title_mm',
    ];
    /**
     * Return searchable fields
     *
     * @return array
     */
    public function getFieldsSearchable()
    {
        return $this->fieldSearchable;
    }

    /**
     * Configure the Model
     **/
    public function model()
    {
        return Book::class;
    }

    /**
     * @param  array  $search
     *
     * @return Book[]|Collection|int
     */
    public function all($search = [])
    {
        $orderBy = null;
        if (! empty($search['order_by']) && in_array($search['order_by'],
                ['name'])) {
            $orderBy = $search['order_by'];
            unset($search['order_by']);
        }
        $query = $this->allQuery($search);
        if(isset($search['status']) && $search['status'])
        {
            $query->where('status',$search['status']);
        }
        // dd($search);
        $bookRecords = $query->paginate($search['paginate']);

        $bookRecords = $query->orderByDesc('id')->paginate($search['paginate']);

        if (! empty($orderBy)) {
            $sortDescending = ($search['direction'] == 'asc') ? false : true;
            $orderString = '';
            if ($orderBy == 'name') {
                $orderString = 'name';
            }

            $bookRecords = $bookRecords->sortBy($orderString, SORT_REGULAR, $sortDescending);
        }

        return $bookRecords;
    }
       /**
     * @param  array  $input
     *
     * @return mixed
     */
    public function store($input)
    {
        try {
            DB::beginTransaction();

            $book=Book::create($input);


            if(!empty($input['file_url']))
            {
                $filePath = Book::makeAttachment($input['file_url'], Book::ATTACHED_PATH);
                $book->file_url= $filePath;
                $book->save();
            }

            save_logs('Create Book',$book);//call helpers method


            DB::commit();
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(new \Illuminate\Support\MessageBag(['catch_exception'=>$e->getMessage()]));
        }

        return $book;
    }
     /**
     * @param  array  $input
     * @param  int  $id
     *
     * @return Book
     */
    public function update($input, $id)
    {
        try {
            DB::beginTransaction();

            /** @var Book $book */
            $book = $this->findOrFail($id);
            $book->update($input);

            if(!empty($input['file_url']))
            {
                Book::deleteImage($book->file_url);
                $filePath = Book::makeAttachment($input['file_url'], Book::ATTACHED_PATH);

                $book->file_url= $filePath;


            }

            $book->updated_by=Auth::id() == 1?2:Auth::id();
            $book->save();

            save_logs('Update Book',$book);//call helpers method


            DB::commit();

            return $book;
        } catch (Exception $e) {
            throw $e;
            DB::rollBack();
            return redirect()->back()->withErrors(new \Illuminate\Support\MessageBag(['catch_exception'=>$e->getMessage()]));
        }
    }

}
