<?php
namespace App\Repository\Backend;

use App\Models\Post;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Eloquent\Builder;
use App\Repository\Backend\BaseRepository;
use App\Repository\Backend\Interfaces\PostRepositoryInterface;

class PostRepository extends BaseRepository implements PostRepositoryInterface
{
    /**
     * @var array
     */
    protected $fieldSearchable = [
        'title',
        'title_mm',
        'body',
        'body_mm',
        'status',
        'meta_keyword'
    ];
    /**
     * Return searchable fields
     *
     * @return array
     */
    public function getFieldsSearchable()
    {
        return $this->fieldSearchable;
    }

    /**
     * Configure the Model
     **/
    public function model()
    {
        return Post::class;
    }

    /**
     * @param  array  $search
     *
     * @return Post[]|Collection|int
     */
    public function all($search = [])
    {
        $orderBy = null;
        if (! empty($search['order_by']) && in_array($search['order_by'],
                ['name'])) {
            $orderBy = $search['order_by'];
            unset($search['order_by']);
        }
        $query = $this->allQuery($search)->with(['CreatedBy']);

        if(isset($search['status']))
        {
            $query->where('status',$search['status']);
        }
        if(isset($search['slider']))
        {
            $query->where('slider',$search['slider']);
        }
        if(isset($search['menu_item_id']))
        {
            $query->where('menu_item_id',$search['menu_item_id']);
        }
        // $query = $this->applyDynamicSearch($search, $query);

        $postRecords = $query->paginate($search['paginate']);

        $postRecords = $query->orderByDesc('id')->paginate($search['paginate']);

        if (! empty($orderBy)) {
            $sortDescending = ($search['direction'] == 'asc') ? false : true;
            $orderString = '';
            if ($orderBy == 'name') {
                $orderString = 'name';
            }

            $postRecords = $postRecords->sortBy($orderString, SORT_REGULAR, $sortDescending);
        }

        return $postRecords;
    }
       /**
     * @param  array  $input
     *
     * @return mixed
     */
    public function store($input)
    {
        // dd($input);
        try {
            DB::beginTransaction();
            $logs=[];
            /** @var Post $post */
            $post=Post::create($input);

            if(!empty($input['feature_image']))
            {
                $featureImagePath = Post::makeImage($input['feature_image'], Post::FEATURE_PATH);
                $post->feature_image= $featureImagePath;

            }

            if(!empty($input['files']))
            {
                foreach($input['files'] as $file)
                {

                    $filePath = Post::makeAttachment($file, Post::ATTACHED_PATH);

                    $file=$post->files()->create([
                        'file_url'=>$filePath,
                        'column_name'=>'file',
                    ]);
                    array_push($logs,$file);
                }
            }
            $post->save();
            array_push($logs,$post);
            // dd($logs);
            save_logs('Store Post',$logs);
            // dd($post);
            DB::commit();
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(new \Illuminate\Support\MessageBag(['catch_exception'=>$e->getMessage()]));
        }

        return $post;
    }
     /**
     * @param  array  $input
     * @param  int  $id
     *
     * @return Post
     */
    public function update($input, $id)
    {

        try {
            DB::beginTransaction();
            $logs=[];
            $post = Post::findOrFail($id);

            $post->update($input);


            if(!empty($input['feature_image']))
            {
                Post::deleteImage($post->feature_image);
                $featureImagePath = Post::makeImage($input['feature_image'], Post::FEATURE_PATH);
                $post->feature_image= $featureImagePath;

            }

            if(!empty($input['files']))
            {
                foreach($input['files'] as $file)
                {

                    $filePath = Post::makeAttachment($file, Post::ATTACHED_PATH);

                    $file=$post->files()->create([
                        'file_url'=>$filePath,
                        'column_name'=>'file',
                    ]);
                    array_push($logs,$file);
                }
            }

            $post->updated_by=Auth::id() == 1?2:Auth::id();
            $post->save();
            array_push($logs,$post);
            save_logs('Update Post',$logs);
            DB::commit();
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(new \Illuminate\Support\MessageBag(['catch_exception'=>$e->getMessage()]));
        }

        return $post;
    }
    /**
     * @param  array  $search
     * @param  Builder  $query
     *
     * @return Builder
     */
    // public function applyDynamicSearch($search, $query)
    // {

    //     $query->when(! empty($search['search']), function (Builder $query) use ($search) {
    //         $query->orWhereHas('roles', function (Builder $query) use ($search) {
    //             filterByColumns($query, $search['search'], ['name']);
    //         });
    //     });

    //     return $query;
    // }
}
