<?php
namespace App\Repository\Backend;

use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use App\Repository\Backend\BaseRepository;
use App\Repository\Backend\Interfaces\RoleRepositoryInterface;

class RoleRepository extends BaseRepository implements RoleRepositoryInterface
{
    /**
     * @var array
     */
    protected $fieldSearchable = [
        'name',
    ];
    /**
     * Return searchable fields
     *
     * @return array
     */
    public function getFieldsSearchable()
    {
        return $this->fieldSearchable;
    }

    /**
     * Configure the Model
     **/
    public function model()
    {
        return Role::class;
    }

    /**
     * @param  array  $search
     *
     * @return Role[]|Collection|int
     */
    public function all($search = [])
    {
        $orderBy = null;

        if (! empty($search['order_by']) && in_array($search['order_by'],
                ['name'])) {
            $orderBy = $search['order_by'];
            unset($search['order_by']);
        }

        $query = $this->allQuery($search)->with(['permissions']);

        $roleRecords = $query->orderByDesc('id')->paginate($search['paginate']);

        if (! empty($orderBy)) {
            $sortDescending = ($search['direction'] == 'asc') ? false : true;
            $orderString = '';
            if ($orderBy == 'name') {
                $orderString = 'name';
            }

            $roleRecords = $roleRecords->sortBy($orderString, SORT_REGULAR, $sortDescending);
        }

        return $roleRecords;
    }
       /**
     * @param  array  $input
     *
     * @return mixed
     */
    public function store($input)
    {
        try {
            DB::beginTransaction();
            /** @var Role $role */
            $role = Role::create(['name'=>$input['name']]);
            $role->syncPermissions([$input['permissions']]);
            save_logs('Store Role',$role);
            DB::commit();
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(new \Illuminate\Support\MessageBag(['catch_exception'=>$e->getMessage()]));
        }

        return $role;
    }
     /**
     * @param  array  $input
     * @param  int  $id
     *
     * @return Role
     */
    public function update($input, $id)
    {
        /** @var Role $role */
        $role = Role::findOrFail($id);
        try {
            DB::beginTransaction();
            $role->update(['name'=>$input['name']]);
            // $permissions = (empty($input['permissions'])) ? [] : $input['permissions'];
            $role->syncPermissions([$input['permissions']]);
            save_logs('Update Role',$role);
            DB::commit();

            return $this->find($id);
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(new \Illuminate\Support\MessageBag(['catch_exception'=>$e->getMessage()]));
        }
    }
}
