<?php
namespace App\Repository\Backend;

use App\Models\Suggestion;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use Illuminate\Database\Eloquent\Builder;
use App\Repository\Backend\BaseRepository;
use App\Repository\Backend\Interfaces\SuggestionRepositoryInterface;

class SuggestionRepository extends BaseRepository implements SuggestionRepositoryInterface
{
    /**
     * @var array
     */
    protected $fieldSearchable = [
        'name',
        'phone',
        'nrc',
        // 'email',
        'address',
        'message',
    ];
    /**
     * Return searchable fields
     *
     * @return array
     */
    public function getFieldsSearchable()
    {
        return $this->fieldSearchable;
    }

    /**
     * Configure the Model
     **/
    public function model()
    {
        return Suggestion::class;
    }

    /**
     * @param  array  $search
     *
     * @return Suggestion[]|Collection|int
     */
    public function all($search = [])
    {
        $orderBy = null;
        if (! empty($search['order_by']) && in_array($search['order_by'],
                ['name'])) {
            $orderBy = $search['order_by'];
            unset($search['order_by']);
        }
        $query = $this->allQuery($search);

        // $suggestionRecords = $query->orderByDesc('id')->paginate($search['paginate']);
        if(isset($search['dates']) && $search['dates'] != null)
        {
            // dd($search);
            $dateRange = $search['dates'];
            [$startDate,$endDate] = explode('-',$dateRange);
            $startDateTime = date('Y-m-d H:i:s',strtotime($startDate));
            $endDateTime = date('Y-m-d',strtotime($endDate)). ' 23:59:59';
            $query = $query->whereBetween('created_at',[$startDateTime,$endDateTime]);
        }
        $suggestionRecords = $query->paginate($search['paginate']);

        if (! empty($orderBy)) {
            $sortDescending = ($search['direction'] == 'asc') ? false : true;
            $orderString = '';
            if ($orderBy == 'name') {
                $orderString = 'name';
            }

            $suggestionRecords = $suggestionRecords->sortBy($orderString, SORT_REGULAR, $sortDescending);
        }

        return $suggestionRecords;
    }
       /**
     * @param  array  $input
     *
     * @return mixed
     */
    public function store($input)
    {
        try {
            DB::beginTransaction();
            $logs=[];
            $suggestion=Suggestion::create($input);


            // if(!empty($input['files']))
            // {
            //     foreach($input['files'] as $file)
            //     {

            //         $filePath = Suggestion::makeAttachment($file, Suggestion::IMAGE_PATH);

            //         $file=$suggestion->files()->create([
            //             'file_url'=>$filePath,
            //             'column_name'=>'file',
            //         ]);
            //         array_push($logs,$file);
            //     }
            // }
            $suggestion->save();
            array_push($logs,$suggestion);
            Log::channel('daily')->info('Store Suggestion',
                [
                    'causer_id'=>'',
                    'data'=>$logs
                ]
            );
            DB::commit();
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(new \Illuminate\Support\MessageBag(['catch_exception'=>$e->getMessage()]));
        }

        return $suggestion;
    }
     /**
     * @param  array  $input
     * @param  int  $id
     *
     * @return Suggestion
     */
    public function update($input, $id)
    {
        try {
            DB::beginTransaction();

            if (! empty($input['password'])) {
                $input['password'] = Hash::make($input['password']);
            }else{
                $input['password']=$input['old_password'];
            }

            $image = (! empty($input['profile_image'])) ? $input['profile_image'] : null;
            unset($input['profile_image']);

            /** @var Suggestion $suggestion */
            $suggestion = $this->findOrFail($id);
            $suggestion->update($input);
            if (! empty($image)) {
                $suggestion->deleteSuggestionImage(); // delete old image;
                $imagePath = Suggestion::makeImage($image, Suggestion::IMAGE_PATH);
                $suggestion->profile_image= $imagePath;
                $suggestion->save();
            }

            if (! empty($input['role'])) {
                $suggestion->roles()->sync($input['role']);
            }

            DB::commit();

            return $suggestion;
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(new \Illuminate\Support\MessageBag(['catch_exception'=>$e->getMessage()]));
        }
    }

}
