<?php
namespace App\Repository\Backend;

use App\Models\User;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Database\Eloquent\Builder;
use App\Repository\Backend\BaseRepository;
use App\Repository\Backend\Interfaces\UserRepositoryInterface;

class UserRepository extends BaseRepository implements UserRepositoryInterface
{
    /**
     * @var array
     */
    protected $fieldSearchable = [
        'name',
        'email',
    ];
    /**
     * Return searchable fields
     *
     * @return array
     */
    public function getFieldsSearchable()
    {
        return $this->fieldSearchable;
    }

    /**
     * Configure the Model
     **/
    public function model()
    {
        return User::class;
    }

    /**
     * @param  array  $search
     *
     * @return User[]|Collection|int
     */
    public function all($search = [])
    {
        $orderBy = null;
        if (! empty($search['order_by']) && in_array($search['order_by'],
                ['name'])) {
            $orderBy = $search['order_by'];
            unset($search['order_by']);
        }
        $query = $this->allQuery($search);
        $query = $this->applyDynamicSearch($search, $query);
        if(isset($search['status']) && $search['status'])
        {
            $query->where('status',$search['status']);
        }
        // dd($search);
        // $userRecords = $query->paginate($search['paginate']);

        $userRecords = $query->where('id','<>',1)->orderByDesc('id')->paginate($search['paginate']);

        if (! empty($orderBy)) {
            $sortDescending = ($search['direction'] == 'asc') ? false : true;
            $orderString = '';
            if ($orderBy == 'name') {
                $orderString = 'name';
            }

            $userRecords = $userRecords->sortBy($orderString, SORT_REGULAR, $sortDescending);
        }

        return $userRecords;
    }
       /**
     * @param  array  $input
     *
     * @return mixed
     */
    public function store($input)
    {
        try {
            DB::beginTransaction();
            /** @var User $user */
            $logs=[];
            $plainPassword=$input['password'];
            $input['password']=Hash::make($plainPassword);
            $user=User::create($input);

            if(!empty($input['role']))
            {
                $user_role=$user->roles()->sync([$input['role']]);
                array_push($logs,$user_role);
            }

            if(!empty($input['profile_image']))
            {
                $imagePath = User::makeImage($input['profile_image'], User::IMAGE_PATH);
                $user->profile_image= $imagePath;
                $user->save();
            }
            array_push($logs,$user);
            save_logs('Store User',$logs);
            DB::commit();
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(new \Illuminate\Support\MessageBag(['catch_exception'=>$e->getMessage()]));
        }

        return $user;
    }
     /**
     * @param  array  $input
     * @param  int  $id
     *
     * @return User
     */
    public function update($input, $id)
    {
        // dd($input);
        try {
            DB::beginTransaction();
            $logs=[];
            if (! empty($input['password'])) {
                $input['password'] = Hash::make($input['password']);
            }else{
                $input['password']=$input['old_password'];
            }

            $image = (! empty($input['profile_image'])) ? $input['profile_image'] : null;
            unset($input['profile_image']);

            /** @var User $user */
            $user = $this->findOrFail($id);
            $user->update($input);
            if (! empty($image)) {
                $user->deleteUserImage(); // delete old image;
                $imagePath = User::makeImage($image, User::IMAGE_PATH);
                $user->profile_image= $imagePath;

            }

            if (! empty($input['role'])) {
                $user_role=$user->roles()->sync($input['role']);
                array_push($logs,$user_role);
            }
            $user->updated_by=Auth::id() == 1?2:Auth::id();
            $user->save();
            array_push($logs,$user);
            save_logs('Update User',$logs);
            DB::commit();

            return $user;
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(new \Illuminate\Support\MessageBag(['catch_exception'=>$e->getMessage()]));
        }
    }
    /**
     * @param  array  $search
     * @param  Builder  $query
     *
     * @return Builder
     */
    public function applyDynamicSearch($search, $query)
    {
        $query->when(! empty($search['search']), function (Builder $query) use ($search) {
            $query->orWhereHas('roles', function (Builder $query) use ($search) {
                filterByColumns($query, $search['search'], ['name']);
            });
        });

        return $query;
    }
}
